#!/usr/bin/env python3
"""
Script to directly call Unitpay payout API
"""

import sys
import os
from decimal import Decimal

# Add the app directory to the path
sys.path.insert(0, os.path.join(os.path.dirname(__file__), 'app'))

from app.services.providers.unitpay_payout_provider import UnitpayPayoutProvider
import secrets
import string

def generate_txnid():
    """Generate a transaction ID"""
    return f"POUT{''.join(secrets.choice(string.ascii_uppercase + string.digits) for _ in range(10))}"

def generate_reference_id():
    """Generate a reference ID in format pay + 8 digits"""
    random_suffix = ''.join(secrets.choice(string.digits) for _ in range(8))
    return f"pay{random_suffix}"

def main():
    amount = Decimal("23970")
    mobile = "9976845230"  # Using default mobile, update if needed
    
    # Beneficiary details
    bene_name = "Account Holder"  # Update with actual beneficiary name if needed
    bene_ifsc = "PUNB0871700"
    bene_acc_no = "8717001700045668"
    
    # Initialize provider
    provider = UnitpayPayoutProvider()
    
    # Process 4 transactions
    for i in range(1, 5):
        # Generate transaction IDs for each transaction
        txnid = generate_txnid()
        user_reference_id = generate_reference_id()
        
        print(f"\n{'='*60}")
        print(f"Unitpay Payout API Call - Transaction {i}/3")
        print(f"{'='*60}")
        print(f"Amount: ₹{amount}")
        print(f"Beneficiary Name: {bene_name}")
        print(f"Account Number: {bene_acc_no}")
        print(f"IFSC: {bene_ifsc}")
        print(f"Mobile: {mobile}")
        print(f"Transaction ID: {txnid}")
        print(f"Reference ID: {user_reference_id}")
        print(f"{'='*60}\n")
        
        try:
            # Make the payout call
            print(f"Calling Unitpay payout API for transaction {i}...")
            response = provider.create_payout(
                txnid=txnid,
                amount=amount,
                bene_name=bene_name,
                bene_ifsc=bene_ifsc,
                bene_acc_no=bene_acc_no,
                user_reference_id=user_reference_id,
                beneficiary_mobile=mobile
            )
            
            # Display response
            print(f"\n{'='*60}")
            print(f"Response - Transaction {i}:")
            print(f"{'='*60}")
            print(f"Status: {response.status}")
            print(f"Message: {response.message}")
            print(f"Reference ID: {response.reference_id}")
            print(f"Transaction ID: {response.txnid}")
            print(f"Amount: {response.amount}")
            
            if response.raw_response:
                print(f"\nRaw Response:")
                import json
                print(json.dumps(response.raw_response, indent=2))
            
            print(f"{'='*60}\n")
            
            if response.status == "pending":
                print(f"✓ Transaction {i} accepted and pending. Status will be updated via webhook.")
            elif response.status == "failed":
                print(f"✗ Transaction {i} failed.")
            elif response.status == "error":
                print(f"✗ Error occurred while processing transaction {i}.")
            
        except Exception as e:
            print(f"\n✗ Error in transaction {i}: {str(e)}")
            import traceback
            traceback.print_exc()
            continue  # Continue with next transaction even if one fails
        
        # Small delay between transactions
        import time
        if i < 3:
            time.sleep(1)
    
    print(f"\n{'='*60}")
    print("All 4 transactions completed!")
    print(f"{'='*60}\n")

if __name__ == "__main__":
    main()

