# Race Condition Analysis - Wallet Transactions

## Critical Issue Found: Race Condition in Wallet Balance Updates

### Problem Summary

The `process_transaction` method in `app/services/wallet_service.py` has a **classic read-modify-write race condition** that causes wallet balance inconsistencies when multiple transactions are processed concurrently.

### Root Cause

**File**: `app/services/wallet_service.py`  
**Method**: `process_transaction()` (lines 83-165)

**The Problem Flow**:
1. **Line 118**: Gets wallet without locking: `wallet = self.get_or_create_wallet(user_id, db)`
2. **Line 129**: Reads current balance: `opening_balance = wallet.main`
3. **Line 135/140**: Calculates new balance
4. **Line 144**: Updates wallet balance: `self.wallet_repo.update(str(wallet.id), main=closing_balance)`

**The Race Condition**:
- Between step 2 (reading balance) and step 4 (updating balance), another concurrent transaction can modify the wallet balance
- Both transactions read the same opening balance, calculate different closing balances, and overwrite each other's updates
- This causes the wallet balance to become incorrect

### Evidence from Database

#### Example 1: Concurrent Transactions with Same TxnID
**Transaction**: `PYT5YQRD45BHST2`
- **Debit** created at `19:36:00.193497` with opening balance `29942.18`
- **Credit (refund)** created at `19:36:00.575546` with opening balance `28922.18`
- **Time difference**: 0.382 seconds
- **Issue**: The refund reads the correct balance, but other concurrent transactions may have read stale balances

#### Example 2: Transactions with Mismatched Opening/Closing Balances
Found **20+ transactions** where the opening balance doesn't match the previous transaction's closing balance, indicating concurrent modifications.

#### Example 3: Multiple Transactions Within Same Second
Many transactions are created within milliseconds of each other, all reading potentially stale wallet balances.

### Impact

1. **Balance Inconsistencies**: Wallet balance doesn't match the sum of all transactions
2. **Lost Updates**: Concurrent transactions overwrite each other's balance updates
3. **Incorrect Opening Balances**: Transactions record incorrect opening balances
4. **Data Integrity Issues**: The transaction chain is broken, making reconciliation impossible

### Current State for User `3c70dd22-df28-4a50-a034-4a378507c5bf`

- **Actual Wallet Balance**: ₹30,962.18
- **Calculated from Transactions**: ₹3,247.76
- **Difference**: ₹27,714.42 (EXTRA in wallet)
- **Broken Transaction Chains**: 20+ transactions with mismatched balances

### Solution: Implement Row-Level Locking

Use PostgreSQL's `SELECT FOR UPDATE` to lock the wallet row during the transaction processing.

**Required Changes**:

1. **Modify `WalletRepository.get_by_user_id()`** to support locking:
   ```python
   def get_by_user_id(self, user_id: str, for_update: bool = False) -> Optional[Wallet]:
       query = self.db.query(self.model).filter(self.model.user_id == user_id)
       if for_update:
           query = query.with_for_update()
       return query.first()
   ```

2. **Modify `WalletService.process_transaction()`** to use locking:
   ```python
   # Get or create wallet WITH LOCK
   wallet = self.get_or_create_wallet(user_id, db, for_update=True)
   ```

3. **Ensure database transaction isolation**: The entire operation should be within a single database transaction.

### Additional Recommendations

1. **Add Database Constraints**: Ensure wallet balance consistency
2. **Implement Reconciliation Job**: Periodically check and fix balance discrepancies
3. **Add Audit Logging**: Log all wallet balance changes for debugging
4. **Consider Optimistic Locking**: Use version numbers to detect concurrent modifications
5. **Add Transaction Retry Logic**: Retry on deadlock/conflict errors

### Priority

**CRITICAL** - This issue causes financial discrepancies and must be fixed immediately.

