"""
Script to compress KYC documents for a specific KYC ID
"""
import os
import sys
import asyncio
from pathlib import Path

# Add parent directory to path to import app modules
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.core.database import SessionLocal
from app.repositories.kyc_doc_repository import KYCDocRepository
from app.services.document_compression_service import DocumentCompressionService
from uuid import UUID


def compress_kyc_documents(kyc_id: str = None):
    """
    Compress all KYC documents
    
    Args:
        kyc_id: Optional KYC ID as string. If None, compresses all documents
    """
    db = SessionLocal()
    try:
        # Get KYC documents
        doc_repo = KYCDocRepository(db)
        
        if kyc_id:
            # Get documents for specific KYC ID
            kyc_uuid = UUID(kyc_id)
            docs = doc_repo.get_by_kyc_id(str(kyc_uuid))
            print(f"Found {len(docs)} document(s) for KYC ID: {kyc_id}\n")
        else:
            # Get all documents (using a very large limit to get all)
            from app.models.kyc_doc import KYCDoc
            docs = db.query(KYCDoc).all()
            print(f"Found {len(docs)} total document(s) to compress\n")
        
        if not docs:
            print("No documents found")
            return
        
        # Initialize compression service
        compression_service = DocumentCompressionService()
        
        # Base upload directory
        UPLOAD_DIR = "uploads/kyc_docs"
        os.makedirs(UPLOAD_DIR, exist_ok=True)
        
        total_docs = len(docs)
        processed_count = 0
        
        for idx, doc in enumerate(docs, 1):
            print(f"\n[{idx}/{total_docs}] Processing document: {doc.document_type} (ID: {doc.id})")
            
            # Process front file
            if doc.front_file_path:
                front_path = doc.front_file_path.replace("static/kyc_docs/", "")
                front_full_path = os.path.join(UPLOAD_DIR, front_path)
                
                if os.path.exists(front_full_path):
                    print(f"  Compressing front file: {front_path}")
                    
                    # Read original file
                    with open(front_full_path, "rb") as f:
                        original_content = f.read()
                    
                    original_size = len(original_content)
                    
                    # Compress
                    try:
                        compressed_content, mime_type = compression_service.compress_document(
                            file_content=original_content,
                            mime_type=doc.mime_type,
                            filename=doc.front_file_name
                        )
                        
                        compressed_size = len(compressed_content)
                        reduction = ((original_size - compressed_size) / original_size * 100) if original_size > 0 else 0
                        
                        print(f"    Original: {original_size} bytes ({doc.mime_type})")
                        print(f"    Compressed: {compressed_size} bytes ({mime_type}) ({reduction:.1f}% reduction)")
                        
                        # Determine new filename based on mime type
                        if mime_type == 'image/webp' and not front_path.endswith('.webp'):
                            # Change extension to .webp
                            base_name = os.path.splitext(front_path)[0]
                            new_front_path = base_name + '.webp'
                            new_front_full_path = os.path.join(UPLOAD_DIR, new_front_path)
                            
                            # Save new WebP file
                            with open(new_front_full_path, "wb") as f:
                                f.write(compressed_content)
                            
                            # Delete old file if it's different
                            if front_full_path != new_front_full_path and os.path.exists(front_full_path):
                                os.remove(front_full_path)
                            
                            # Update database with new path and mime type
                            doc_repo.update(
                                str(doc.id),
                                front_file_path=f"static/kyc_docs/{new_front_path}",
                                file_size=str(compressed_size),
                                mime_type=mime_type
                            )
                        else:
                            # Save compressed file (overwrite original)
                            with open(front_full_path, "wb") as f:
                                f.write(compressed_content)
                            
                            # Update database with new file size and mime type
                            doc_repo.update(
                                str(doc.id),
                                file_size=str(compressed_size),
                                mime_type=mime_type
                            )
                        
                        print(f"    ✓ Front file compressed and updated")
                        processed_count += 1
                        
                    except Exception as e:
                        print(f"    ✗ Error compressing front file: {str(e)}")
                else:
                    print(f"    ✗ Front file not found: {front_full_path}")
            
            # Process back file
            if doc.back_file_path:
                back_path = doc.back_file_path.replace("static/kyc_docs/", "")
                back_full_path = os.path.join(UPLOAD_DIR, back_path)
                
                if os.path.exists(back_full_path):
                    print(f"  Compressing back file: {back_path}")
                    
                    # Read original file
                    with open(back_full_path, "rb") as f:
                        original_content = f.read()
                    
                    original_size = len(original_content)
                    
                    # Compress
                    try:
                        compressed_content, mime_type = compression_service.compress_document(
                            file_content=original_content,
                            mime_type=doc.mime_type,
                            filename=doc.back_file_name
                        )
                        
                        compressed_size = len(compressed_content)
                        reduction = ((original_size - compressed_size) / original_size * 100) if original_size > 0 else 0
                        
                        print(f"    Original: {original_size} bytes ({doc.mime_type})")
                        print(f"    Compressed: {compressed_size} bytes ({mime_type}) ({reduction:.1f}% reduction)")
                        
                        # Determine new filename based on mime type
                        if mime_type == 'image/webp' and not back_path.endswith('.webp'):
                            # Change extension to .webp
                            base_name = os.path.splitext(back_path)[0]
                            new_back_path = base_name + '.webp'
                            new_back_full_path = os.path.join(UPLOAD_DIR, new_back_path)
                            
                            # Save new WebP file
                            with open(new_back_full_path, "wb") as f:
                                f.write(compressed_content)
                            
                            # Delete old file if it's different
                            if back_full_path != new_back_full_path and os.path.exists(back_full_path):
                                os.remove(back_full_path)
                            
                            # Update database with new path and mime type
                            doc_repo.update(
                                str(doc.id),
                                back_file_path=f"static/kyc_docs/{new_back_path}",
                                file_size=str(compressed_size),
                                mime_type=mime_type
                            )
                        else:
                            # Save compressed file (overwrite original)
                            with open(back_full_path, "wb") as f:
                                f.write(compressed_content)
                            
                            # Update database with new file size and mime type
                            doc_repo.update(
                                str(doc.id),
                                file_size=str(compressed_size),
                                mime_type=mime_type
                            )
                        
                        print(f"    ✓ Back file compressed and updated")
                        processed_count += 1
                        
                    except Exception as e:
                        print(f"    ✗ Error compressing back file: {str(e)}")
                else:
                    print(f"    ✗ Back file not found: {back_full_path}")
        
        print(f"\n✓ Completed processing {len(docs)} document(s)")
        print(f"✓ Successfully compressed {processed_count} file(s)")
        
    except Exception as e:
        print(f"Error: {str(e)}")
        import traceback
        traceback.print_exc()
    finally:
        db.close()


if __name__ == "__main__":
    import sys
    
    # Check if KYC ID is provided as command line argument
    if len(sys.argv) > 1:
        kyc_id = sys.argv[1]
        print(f"Compressing KYC documents for KYC ID: {kyc_id}\n")
        compress_kyc_documents(kyc_id)
    else:
        print("Compressing ALL KYC documents in the database\n")
        compress_kyc_documents()

