"""
Script to convert a specific WebP file back to PNG
"""
import os
import sys
from pathlib import Path

# Add parent directory to path to import app modules
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.core.database import SessionLocal
from app.repositories.kyc_doc_repository import KYCDocRepository
from app.models.kyc_doc import KYCDoc
from PIL import Image
import io


def convert_webp_to_png(filename: str):
    """
    Find and convert a specific WebP file to PNG
    
    Args:
        filename: The filename to convert (e.g., '3c70dd22-df28-4a50-a034-4a378507c5bf_1766738489_front_11111.webp')
    """
    db = SessionLocal()
    try:
        # Find document by file path
        doc_repo = KYCDocRepository(db)
        
        # Search for document with this filename in front_file_path or back_file_path
        docs = db.query(KYCDoc).filter(
            (KYCDoc.front_file_path.like(f'%{filename}')) |
            (KYCDoc.back_file_path.like(f'%{filename}'))
        ).all()
        
        if not docs:
            print(f"No document found with filename: {filename}")
            return
        
        print(f"Found {len(docs)} document(s) with filename: {filename}\n")
        
        # Base upload directory
        UPLOAD_DIR = "uploads/kyc_docs"
        os.makedirs(UPLOAD_DIR, exist_ok=True)
        
        for doc in docs:
            print(f"Processing document: {doc.document_type} (ID: {doc.id})")
            
            # Check if it's a front file
            if doc.front_file_path and filename in doc.front_file_path:
                front_path = doc.front_file_path.replace("static/kyc_docs/", "")
                front_full_path = os.path.join(UPLOAD_DIR, front_path)
                
                if os.path.exists(front_full_path):
                    print(f"  Converting front file: {front_path}")
                    
                    # Read WebP file
                    with open(front_full_path, "rb") as f:
                        webp_content = f.read()
                    
                    webp_size = len(webp_content)
                    
                    try:
                        # Open WebP image
                        img = Image.open(io.BytesIO(webp_content))
                        
                        # Convert to PNG
                        png_buffer = io.BytesIO()
                        img.save(png_buffer, format='PNG', optimize=True)
                        png_content = png_buffer.getvalue()
                        png_size = len(png_content)
                        
                        print(f"    WebP size: {webp_size} bytes")
                        print(f"    PNG size: {png_size} bytes")
                        
                        # Create new PNG filename
                        base_name = os.path.splitext(front_path)[0]
                        new_front_path = base_name + '.png'
                        new_front_full_path = os.path.join(UPLOAD_DIR, new_front_path)
                        
                        # Save PNG file
                        with open(new_front_full_path, "wb") as f:
                            f.write(png_content)
                        
                        # Delete old WebP file
                        if front_full_path != new_front_full_path and os.path.exists(front_full_path):
                            os.remove(front_full_path)
                        
                        # Update database
                        doc_repo.update(
                            str(doc.id),
                            front_file_path=f"static/kyc_docs/{new_front_path}",
                            file_size=str(png_size),
                            mime_type="image/png"
                        )
                        
                        print(f"    ✓ Front file converted to PNG and updated")
                        
                    except Exception as e:
                        print(f"    ✗ Error converting front file: {str(e)}")
                        import traceback
                        traceback.print_exc()
                else:
                    print(f"    ✗ Front file not found: {front_full_path}")
            
            # Check if it's a back file
            if doc.back_file_path and filename in doc.back_file_path:
                back_path = doc.back_file_path.replace("static/kyc_docs/", "")
                back_full_path = os.path.join(UPLOAD_DIR, back_path)
                
                if os.path.exists(back_full_path):
                    print(f"  Converting back file: {back_path}")
                    
                    # Read WebP file
                    with open(back_full_path, "rb") as f:
                        webp_content = f.read()
                    
                    webp_size = len(webp_content)
                    
                    try:
                        # Open WebP image
                        img = Image.open(io.BytesIO(webp_content))
                        
                        # Convert to PNG
                        png_buffer = io.BytesIO()
                        img.save(png_buffer, format='PNG', optimize=True)
                        png_content = png_buffer.getvalue()
                        png_size = len(png_content)
                        
                        print(f"    WebP size: {webp_size} bytes")
                        print(f"    PNG size: {png_size} bytes")
                        
                        # Create new PNG filename
                        base_name = os.path.splitext(back_path)[0]
                        new_back_path = base_name + '.png'
                        new_back_full_path = os.path.join(UPLOAD_DIR, new_back_path)
                        
                        # Save PNG file
                        with open(new_back_full_path, "wb") as f:
                            f.write(png_content)
                        
                        # Delete old WebP file
                        if back_full_path != new_back_full_path and os.path.exists(back_full_path):
                            os.remove(back_full_path)
                        
                        # Update database
                        doc_repo.update(
                            str(doc.id),
                            back_file_path=f"static/kyc_docs/{new_back_path}",
                            file_size=str(png_size),
                            mime_type="image/png"
                        )
                        
                        print(f"    ✓ Back file converted to PNG and updated")
                        
                    except Exception as e:
                        print(f"    ✗ Error converting back file: {str(e)}")
                        import traceback
                        traceback.print_exc()
                else:
                    print(f"    ✗ Back file not found: {back_full_path}")
        
        print(f"\n✓ Completed processing")
        
    except Exception as e:
        print(f"Error: {str(e)}")
        import traceback
        traceback.print_exc()
    finally:
        db.close()


if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python scripts/convert_webp_to_png.py <filename>")
        print("Example: python scripts/convert_webp_to_png.py 3c70dd22-df28-4a50-a034-4a378507c5bf_1766738489_front_11111.webp")
        sys.exit(1)
    
    filename = sys.argv[1]
    print(f"Converting WebP file to PNG: {filename}\n")
    convert_webp_to_png(filename)

