#!/usr/bin/env python3
"""
Script to create a dummy admin account
Usage: docker compose exec api python scripts/create_admin.py
"""

import sys
import os

# Add parent directory to path
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

import bcrypt
from sqlalchemy.orm import Session
from app.core.database import SessionLocal
from app.repositories.admin_repository import AdminRepository
from app.models.admin import Admin


def create_dummy_admin():
    """Create a dummy admin account"""
    db: Session = SessionLocal()
    
    try:
        admin_repo = AdminRepository(db)
        
        # Check if admin already exists
        existing_admin = admin_repo.get_by_email("admin@fintech.com")
        if existing_admin:
            print("⚠️  Admin with email 'admin@fintech.com' already exists!")
            print(f"   Admin ID: {existing_admin.id}")
            return
        
        # Hash password with bcrypt directly
        password = "admin123"
        password_hash = bcrypt.hashpw(password.encode('utf-8'), bcrypt.gensalt(rounds=12)).decode('utf-8')
        
        # Create admin
        admin = Admin(
            email="admin@fintech.com",
            password_hash=password_hash,
            full_name="Dummy Admin",
            is_active=True
        )
        db.add(admin)
        db.commit()
        db.refresh(admin)
        
        print("✅ Dummy admin account created successfully!")
        print(f"   Email: {admin.email}")
        print(f"   Name: {admin.full_name}")
        print(f"   ID: {admin.id}")
        print(f"   Password: admin123")
        print("\n⚠️  WARNING: Change this password in production!")
        
    except Exception as e:
        print(f"❌ Error creating admin: {e}")
        db.rollback()
        raise
    finally:
        db.close()


if __name__ == "__main__":
    create_dummy_admin()
