"""
Script to fix a specific WebP file by re-compressing it
"""
import os
import sys
from pathlib import Path

# Add parent directory to path to import app modules
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.core.database import SessionLocal
from app.repositories.kyc_doc_repository import KYCDocRepository
from app.services.document_compression_service import DocumentCompressionService
from app.models.kyc_doc import KYCDoc


def fix_webp_file(filename: str):
    """
    Find and re-compress a specific WebP file
    
    Args:
        filename: The filename to fix (e.g., '3c70dd22-df28-4a50-a034-4a378507c5bf_1766738489_front_11111.webp')
    """
    db = SessionLocal()
    try:
        # Find document by file path
        doc_repo = KYCDocRepository(db)
        
        # Search for document with this filename in front_file_path or back_file_path
        docs = db.query(KYCDoc).filter(
            (KYCDoc.front_file_path.like(f'%{filename}')) |
            (KYCDoc.back_file_path.like(f'%{filename}'))
        ).all()
        
        if not docs:
            print(f"No document found with filename: {filename}")
            return
        
        print(f"Found {len(docs)} document(s) with filename: {filename}\n")
        
        # Initialize compression service
        compression_service = DocumentCompressionService()
        
        # Base upload directory
        UPLOAD_DIR = "uploads/kyc_docs"
        os.makedirs(UPLOAD_DIR, exist_ok=True)
        
        for doc in docs:
            print(f"Processing document: {doc.document_type} (ID: {doc.id})")
            
            # Check if it's a front file
            if doc.front_file_path and filename in doc.front_file_path:
                front_path = doc.front_file_path.replace("static/kyc_docs/", "")
                front_full_path = os.path.join(UPLOAD_DIR, front_path)
                
                if os.path.exists(front_full_path):
                    print(f"  Re-compressing front file: {front_path}")
                    
                    # Read original file
                    with open(front_full_path, "rb") as f:
                        original_content = f.read()
                    
                    original_size = len(original_content)
                    
                    # Re-compress to WebP
                    try:
                        compressed_content, mime_type = compression_service.compress_document(
                            file_content=original_content,
                            mime_type="image/webp",  # Force WebP
                            filename=doc.front_file_name
                        )
                        
                        compressed_size = len(compressed_content)
                        reduction = ((original_size - compressed_size) / original_size * 100) if original_size > 0 else 0
                        
                        print(f"    Original: {original_size} bytes")
                        print(f"    Compressed: {compressed_size} bytes ({reduction:.1f}% reduction)")
                        
                        # Ensure filename has .webp extension
                        if not front_path.endswith('.webp'):
                            base_name = os.path.splitext(front_path)[0]
                            new_front_path = base_name + '.webp'
                            new_front_full_path = os.path.join(UPLOAD_DIR, new_front_path)
                            
                            # Save new WebP file
                            with open(new_front_full_path, "wb") as f:
                                f.write(compressed_content)
                            
                            # Delete old file if different
                            if front_full_path != new_front_full_path and os.path.exists(front_full_path):
                                os.remove(front_full_path)
                            
                            # Update database
                            doc_repo.update(
                                str(doc.id),
                                front_file_path=f"static/kyc_docs/{new_front_path}",
                                file_size=str(compressed_size),
                                mime_type="image/webp"
                            )
                        else:
                            # Overwrite existing file
                            with open(front_full_path, "wb") as f:
                                f.write(compressed_content)
                            
                            # Update database
                            doc_repo.update(
                                str(doc.id),
                                file_size=str(compressed_size),
                                mime_type="image/webp"
                            )
                        
                        print(f"    ✓ Front file re-compressed and updated")
                        
                    except Exception as e:
                        print(f"    ✗ Error re-compressing front file: {str(e)}")
                        import traceback
                        traceback.print_exc()
                else:
                    print(f"    ✗ Front file not found: {front_full_path}")
            
            # Check if it's a back file
            if doc.back_file_path and filename in doc.back_file_path:
                back_path = doc.back_file_path.replace("static/kyc_docs/", "")
                back_full_path = os.path.join(UPLOAD_DIR, back_path)
                
                if os.path.exists(back_full_path):
                    print(f"  Re-compressing back file: {back_path}")
                    
                    # Read original file
                    with open(back_full_path, "rb") as f:
                        original_content = f.read()
                    
                    original_size = len(original_content)
                    
                    # Re-compress to WebP
                    try:
                        compressed_content, mime_type = compression_service.compress_document(
                            file_content=original_content,
                            mime_type="image/webp",  # Force WebP
                            filename=doc.back_file_name
                        )
                        
                        compressed_size = len(compressed_content)
                        reduction = ((original_size - compressed_size) / original_size * 100) if original_size > 0 else 0
                        
                        print(f"    Original: {original_size} bytes")
                        print(f"    Compressed: {compressed_size} bytes ({reduction:.1f}% reduction)")
                        
                        # Ensure filename has .webp extension
                        if not back_path.endswith('.webp'):
                            base_name = os.path.splitext(back_path)[0]
                            new_back_path = base_name + '.webp'
                            new_back_full_path = os.path.join(UPLOAD_DIR, new_back_path)
                            
                            # Save new WebP file
                            with open(new_back_full_path, "wb") as f:
                                f.write(compressed_content)
                            
                            # Delete old file if different
                            if back_full_path != new_back_full_path and os.path.exists(back_full_path):
                                os.remove(back_full_path)
                            
                            # Update database
                            doc_repo.update(
                                str(doc.id),
                                back_file_path=f"static/kyc_docs/{new_back_path}",
                                file_size=str(compressed_size),
                                mime_type="image/webp"
                            )
                        else:
                            # Overwrite existing file
                            with open(back_full_path, "wb") as f:
                                f.write(compressed_content)
                            
                            # Update database
                            doc_repo.update(
                                str(doc.id),
                                file_size=str(compressed_size),
                                mime_type="image/webp"
                            )
                        
                        print(f"    ✓ Back file re-compressed and updated")
                        
                    except Exception as e:
                        print(f"    ✗ Error re-compressing back file: {str(e)}")
                        import traceback
                        traceback.print_exc()
                else:
                    print(f"    ✗ Back file not found: {back_full_path}")
        
        print(f"\n✓ Completed processing")
        
    except Exception as e:
        print(f"Error: {str(e)}")
        import traceback
        traceback.print_exc()
    finally:
        db.close()


if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python scripts/fix_specific_webp.py <filename>")
        print("Example: python scripts/fix_specific_webp.py 3c70dd22-df28-4a50-a034-4a378507c5bf_1766738489_front_11111.webp")
        sys.exit(1)
    
    filename = sys.argv[1]
    print(f"Fixing WebP file: {filename}\n")
    fix_webp_file(filename)

