"""
Test script to convert one file to WebP using the working method
"""
from PIL import Image
import sys
import os
from pathlib import Path

# Add parent directory to path to import app modules
sys.path.insert(0, str(Path(__file__).parent.parent))

from app.core.database import SessionLocal
from app.repositories.kyc_doc_repository import KYCDocRepository
from app.models.kyc_doc import KYCDoc


def convert_to_webp(input_path, output_path, quality=80):
    """Convert image to WebP using the working method"""
    # Open image safely
    with Image.open(input_path) as img:
        # Convert to RGB (VERY IMPORTANT)
        if img.mode in ("RGBA", "P"):
            img = img.convert("RGB")
        
        # Save as real WebP binary
        img.save(output_path, format="WEBP", quality=quality, method=6)
    
    print("✅ WebP created successfully:", output_path)


def test_convert_one_file(filename: str):
    """
    Test converting one file to WebP
    
    Args:
        filename: The filename to convert (e.g., '3c70dd22-df28-4a50-a034-4a378507c5bf_1766738489_front_11111.png')
    """
    db = SessionLocal()
    try:
        # Find document by file path
        doc_repo = KYCDocRepository(db)
        
        # Search for document with this filename
        docs = db.query(KYCDoc).filter(
            (KYCDoc.front_file_path.like(f'%{filename}')) |
            (KYCDoc.back_file_path.like(f'%{filename}'))
        ).all()
        
        if not docs:
            print(f"No document found with filename: {filename}")
            return
        
        print(f"Found {len(docs)} document(s) with filename: {filename}\n")
        
        # Base upload directory
        UPLOAD_DIR = "uploads/kyc_docs"
        os.makedirs(UPLOAD_DIR, exist_ok=True)
        
        for doc in docs:
            print(f"Processing document: {doc.document_type} (ID: {doc.id})")
            
            # Check if it's a front file
            if doc.front_file_path and filename in doc.front_file_path:
                front_path = doc.front_file_path.replace("static/kyc_docs/", "")
                front_full_path = os.path.join(UPLOAD_DIR, front_path)
                
                if os.path.exists(front_full_path):
                    print(f"  Converting front file: {front_path}")
                    
                    # Get original file size
                    original_size = os.path.getsize(front_full_path)
                    print(f"    Original size: {original_size} bytes")
                    
                    # Create WebP filename
                    base_name = os.path.splitext(front_path)[0]
                    webp_path = base_name + '.webp'
                    webp_full_path = os.path.join(UPLOAD_DIR, webp_path)
                    
                    try:
                        # Convert to WebP using the working method
                        convert_to_webp(front_full_path, webp_full_path, quality=80)
                        
                        # Get new file size
                        webp_size = os.path.getsize(webp_full_path)
                        reduction = ((original_size - webp_size) / original_size * 100) if original_size > 0 else 0
                        
                        print(f"    WebP size: {webp_size} bytes ({reduction:.1f}% reduction)")
                        
                        # Verify the WebP file is valid
                        with Image.open(webp_full_path) as test_img:
                            print(f"    ✓ Valid WebP: {test_img.size}, mode: {test_img.mode}")
                        
                        # Update database
                        doc_repo.update(
                            str(doc.id),
                            front_file_path=f"static/kyc_docs/{webp_path}",
                            file_size=str(webp_size),
                            mime_type="image/webp"
                        )
                        
                        # Delete old PNG file
                        if front_full_path != webp_full_path and os.path.exists(front_full_path):
                            os.remove(front_full_path)
                            print(f"    ✓ Old file deleted")
                        
                        print(f"    ✓ Front file converted to WebP and database updated")
                        
                    except Exception as e:
                        print(f"    ✗ Error converting front file: {str(e)}")
                        import traceback
                        traceback.print_exc()
                else:
                    print(f"    ✗ Front file not found: {front_full_path}")
            
            # Check if it's a back file
            if doc.back_file_path and filename in doc.back_file_path:
                back_path = doc.back_file_path.replace("static/kyc_docs/", "")
                back_full_path = os.path.join(UPLOAD_DIR, back_path)
                
                if os.path.exists(back_full_path):
                    print(f"  Converting back file: {back_path}")
                    
                    # Get original file size
                    original_size = os.path.getsize(back_full_path)
                    print(f"    Original size: {original_size} bytes")
                    
                    # Create WebP filename
                    base_name = os.path.splitext(back_path)[0]
                    webp_path = base_name + '.webp'
                    webp_full_path = os.path.join(UPLOAD_DIR, webp_path)
                    
                    try:
                        # Convert to WebP using the working method
                        convert_to_webp(back_full_path, webp_full_path, quality=80)
                        
                        # Get new file size
                        webp_size = os.path.getsize(webp_full_path)
                        reduction = ((original_size - webp_size) / original_size * 100) if original_size > 0 else 0
                        
                        print(f"    WebP size: {webp_size} bytes ({reduction:.1f}% reduction)")
                        
                        # Verify the WebP file is valid
                        with Image.open(webp_full_path) as test_img:
                            print(f"    ✓ Valid WebP: {test_img.size}, mode: {test_img.mode}")
                        
                        # Update database
                        doc_repo.update(
                            str(doc.id),
                            back_file_path=f"static/kyc_docs/{webp_path}",
                            file_size=str(webp_size),
                            mime_type="image/webp"
                        )
                        
                        # Delete old file
                        if back_full_path != webp_full_path and os.path.exists(back_full_path):
                            os.remove(back_full_path)
                            print(f"    ✓ Old file deleted")
                        
                        print(f"    ✓ Back file converted to WebP and database updated")
                        
                    except Exception as e:
                        print(f"    ✗ Error converting back file: {str(e)}")
                        import traceback
                        traceback.print_exc()
                else:
                    print(f"    ✗ Back file not found: {back_full_path}")
        
        print(f"\n✓ Test completed")
        
    except Exception as e:
        print(f"Error: {str(e)}")
        import traceback
        traceback.print_exc()
    finally:
        db.close()


if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python scripts/test_webp_conversion.py <filename>")
        print("Example: python scripts/test_webp_conversion.py 3c70dd22-df28-4a50-a034-4a378507c5bf_1766738489_front_11111.png")
        sys.exit(1)
    
    filename = sys.argv[1]
    print(f"Testing WebP conversion for: {filename}\n")
    test_convert_one_file(filename)

